
/*
 * For each town in the game, an instance of this class is created.
 * It holds the data related to a specific town.
 */
class GoalTown
{
	id = 0;
	rate = 0;
	check_timer = 0;
	neighbors = null;
	is_congested = null;
	congestion_goal_id = null;
	prev_enable_neighbours_value = null;

	constructor(town_id)
	{
		this.id = town_id;
		this.rate = 1;
		this.neighbors = [];
		this.is_congested = false;
		this.congestion_goal_id = -1;
		this.prev_enable_neighbours_value = GSController.GetSetting("enable_neighbours");

		local all_towns = GSTownList();

		foreach(t, _ in all_towns)
		{
			/* Skip town if it is the same as this.id */
			if(t == this.id)
				continue;

			/* Add towns with distance less than 100 tiles to neighbor list */
			local distance = GSMap.DistanceManhattan(GSTown.GetLocation(town_id), GSTown.GetLocation(t));
			if(distance < 100)
			{
				neighbors.append( { id = t, dist = distance } );
			}
		}

		/* Sort by distance */
		neighbors.sort(TownDistSorter);

		/* Remove the most distant towns, so that only three neighbors remain */
		while(neighbors.len() > 3)
		{
			neighbors.pop();
		}

		this.UpdateTownText();
	}

	function ManageTown(); // <-- Main function to process town

	// private functions for town management
	function UpdateTownText();
	function SetGrowthRate();
	function SetGoal();
	function CheckCongestion();
}

function GoalTown::ManageTown()
{
	this.CheckCongestion();
	this.SetGrowthRate();
	this.SetGoal();

	// The town text doesn't include any goal specific content and does
	// in general not need to be updated more than when the game starts.
	// However, the enable_neighbours setting can be changed in-game,
	// and in the case that it is changed, the text needs to be updated.
	local curr_setting_value = GSController.GetSetting("enable_neighbours")
	if (this.prev_enable_neighbours_value != curr_setting_value)
	{
		this.prev_enable_neighbours_value = curr_setting_value;
		this.UpdateTownText();
	}
}

function GoalTown::UpdateTownText()
{
	// If neighbours are enabled, display the town neighbours in the town window
	local text = "";
	if (GSController.GetSetting("enable_neighbours") == 1)
	{
		text = GSText(GSText.STR_0_NEIGHBORS_TOWN_GUI + this.neighbors.len());
		foreach(n in this.neighbors)
		{
			text.AddParam(n.id);
		}
		if (this.neighbors.len() > 0)
		{
			// Add info-text that informs players how neighbours work only if there are at least one neighbour.
			text.AddParam(GSText(GSText.STR_NEIGHBOUR_GROWTH_INFO));
		}
	}

	GSTown.SetText(this.id, text);
}

function TownDistSorter(a, b) { 
	return a.dist == b.dist? 1 : (a.dist > b.dist ? 1 : -1); 
}

function GoalTown::SetGrowthRate()
{
	local congestion_policy = GSController.GetSetting("congestion_policy");
	if(congestion_policy == CONGESTION_GROW_STOP && this.is_congested)
	{
		// Congestion stops town from growing
		GSTown.SetGrowthRate(this.id, 365 * 1000); // grow every 1000 years
		return;
	}

	local rating = GSTown.TOWN_RATING_NONE;
	for (local companyID = GSCompany.COMPANY_FIRST; companyID <= GSCompany.COMPANY_LAST; companyID++) {
		if (GSCompany.ResolveCompanyID(companyID) != GSCompany.COMPANY_INVALID) {
			local r = GSTown.GetRating(this.id, companyID);
			if (r > rating) rating = r;
		}
	}

	local rate = 420;
	switch (rating) {
		case GSTown.TOWN_RATING_NONE:        rate = 420; break;
		case GSTown.TOWN_RATING_APPALLING:   rate = 360; break;
		case GSTown.TOWN_RATING_VERY_POOR:   rate = 300; break;
		case GSTown.TOWN_RATING_POOR:        rate = 260; break;
		case GSTown.TOWN_RATING_MEDIOCRE:    rate = 220; break;
		case GSTown.TOWN_RATING_GOOD:        rate = 190; break;
		case GSTown.TOWN_RATING_VERY_GOOD:   rate = 160; break;
		case GSTown.TOWN_RATING_EXCELLENT:   rate = 130; break;
		case GSTown.TOWN_RATING_OUTSTANDING: rate = 100; break;
	}

	if(congestion_policy == CONGESTION_GROW_REDUCED && this.is_congested)
	{
		rate *= 4; // increase time between growth with factor 4
	}

	local town_growth_rate = GSGameSettings.GetValue("economy.town_growth_rate") - 1;
	if (town_growth_rate < 1) town_growth_rate = 1;

	rate = rate >> town_growth_rate;
	rate = rate / (GSTown.GetPopulation(this.id) / 250 + 1);

	if (rate < 1) rate = 1;
	GSTown.SetGrowthRate(this.id, rate);
}

function MaxAsInt(a, b)
{
	return a > b? a.tointeger() : b.tointeger();
}

function GoalTown::SetGoal()
{
	// Check the vehicle congestion of the town
	this.CheckCongestion();

	local population = GSTown.GetPopulation(this.id);
	local location = GSTown.GetLocation(this.id);

	// Neighbour factor:
	local n_percent = GSController.GetSetting("enable_neighbours") == 1?
		this.GetNeighborhoodFactor(population) : 0;
	local n_factor = n_percent > 0? n_percent / 100.0 : 1.0;

	// Difficulty factor:
	local d_percent = GSController.GetSetting("goal_scale_factor");
	local d_factor = d_percent > 0? d_percent / 100.0 : 1.0; // Use 1.0 if setting is missing

	// Compute a combined neighbour + difficulty factor
	local factor = n_factor * d_factor;

	switch (GSGame.GetLandscape()) {
		case GSGame.LT_TEMPERATE:
		case GSGame.LT_TOYLAND:
			GSTown.SetCargoGoal(this.id, GSCargo.TE_PASSENGERS, MaxAsInt(((population / 10) - 5) * factor, 1));
			GSTown.SetCargoGoal(this.id, GSCargo.TE_MAIL,       MaxAsInt(((population / 50) - 10) * factor, 0));
			GSTown.SetCargoGoal(this.id, GSCargo.TE_GOODS,      MaxAsInt(((population / 100) - 10) * factor, 0));
			break;

		case GSGame.LT_ARCTIC:
			GSTown.SetCargoGoal(this.id, GSCargo.TE_PASSENGERS, MaxAsInt(((population / 10) - 5) * factor, 1));
			GSTown.SetCargoGoal(this.id, GSCargo.TE_FOOD,       MaxAsInt(((population / 20) - 5) * factor, 0));
			GSTown.SetCargoGoal(this.id, GSCargo.TE_MAIL,       MaxAsInt(((population / 50) - 10) * factor, 0));
			GSTown.SetCargoGoal(this.id, GSCargo.TE_GOODS,      MaxAsInt(((population / 100) - 20) * factor, 0));
			break;

		case GSGame.LT_TROPIC:
			// non-desert towns have the 'water' requirement added to the food requirement instead.
			// It will not make food requirement kick in earlier, but when it does it will grow quicker.
			local desert_town = GSTile.IsDesertTile(location);
			local water_requirement = MaxAsInt(((population / 50) - 2) * factor, 0);
			GSTown.SetCargoGoal(this.id, GSCargo.TE_PASSENGERS, MaxAsInt(((population / 10) - 5) * factor, 1));
			GSTown.SetCargoGoal(this.id, GSCargo.TE_FOOD,       MaxAsInt(((population / 20) - 5) * factor, 0) + (desert_town? 0 : water_requirement));
			GSTown.SetCargoGoal(this.id, GSCargo.TE_WATER,      desert_town? water_requirement : 0);
			GSTown.SetCargoGoal(this.id, GSCargo.TE_MAIL,       MaxAsInt(((population / 50) - 10) * factor, 0));
			GSTown.SetCargoGoal(this.id, GSCargo.TE_GOODS,      MaxAsInt(((population / 100) - 20) * factor, 0));
			break;
	}
}

function GoalTown::GetNeighborhoodFactor(self_population)
{
	local factor = 100;

	foreach(n in this.neighbors)
	{
		local diff = GSTown.GetPopulation(n.id) - self_population; 
		factor -= diff * 3;
	}

	factor = max(factor, 70);
	factor = min(factor, 500);

	Log.Info(GSTown.GetName(this.id) + " has neighbour factor " + factor, Log.LVL_DEBUG);

	return factor;
}

function GoalTown::VehicleInRectValuator(vehicle, min_x, min_y, max_x, max_y)
{
	local loc = GSVehicle.GetLocation(vehicle);
	local x = GSMap.GetTileX(loc);
	local y = GSMap.GetTileY(loc);

	return x >= min_x && y >= min_y && x <= max_x && y <= max_y;
}
function GoalTown::CheckCongestion()
{
	local congestion_policy = GSController.GetSetting("congestion_policy");

	// don't waste time on finding out congestion if towns don't care
	if(congestion_policy == CONGESTION_DONT_CARE)
	{
		// Remove congestion sign in case the GS setting was changed in-game
		Helper.SetSign(GSTown.GetLocation(this.id), "", true);
		return; 
	}

	// Which area to consider for congestion?
	local center = GSTown.GetLocation(this.id);
	local radius = min(GSTown.GetPopulation(this.id) / 150 + 2, 8);

	local north_corner = Direction.GetTileInDirection(center, Direction.DIR_N, radius);
	local south_corner = Direction.GetTileInDirection(center, Direction.DIR_S, radius);
	local min_x = GSMap.GetTileX(north_corner);
	local min_y = GSMap.GetTileY(north_corner);
	local max_x = GSMap.GetTileX(south_corner);
	local max_y = GSMap.GetTileY(south_corner);

	local square_side = radius * 2 + 1;
	local tile_count = square_side * square_side;

	// Figure out how many road vehicles there are in the town
	local total_vehicles_count = 0;
	local companies = [];
	for(local c = GSCompany.COMPANY_FIRST; c <= GSCompany.COMPANY_LAST; c++)
	{
		if(GSCompany.ResolveCompanyID(c) == GSCompany.COMPANY_INVALID)
			continue;

		local cm = GSCompanyMode(c);

		local vehicles = GSVehicleList();
		vehicles.Valuate(GSVehicle.GetVehicleType);
		vehicles.KeepValue(GSVehicle.VT_ROAD);
		vehicles.Valuate(GoalTown.VehicleInRectValuator, min_x, min_y, max_x, max_y);
		vehicles.KeepValue(1);

		local company_count = vehicles.Count();
		if(company_count > 0)
			companies.append(c);

		total_vehicles_count += company_count;
	}

	// Is the town congested?
	local congestion_limit_factor = GSController.GetSetting("congestion_limit_factor");
	congestion_limit_factor = congestion_limit_factor > 0? congestion_limit_factor / 100.0 : 1.0;
	local congestion_limit = (min(radius * 3 + GSTown.GetPopulation(this.id) / 25, 100) * congestion_limit_factor).tointeger();
	local is_congested = total_vehicles_count > congestion_limit;

	Log.Info("Vehicles in " + GSTown.GetName(this.id) + ": " + total_vehicles_count + "   congestion limit: " + congestion_limit + "  radius: " + radius, Log.LVL_DEBUG);

	// Has the congested state changed since last check?
	if(is_congested != this.is_congested)
	{
		// Remove old goal regardless if we expect there to be any
		GSGoal.Remove(this.congestion_goal_id);
		this.congestion_goal_id = -1;

		if(is_congested)
		{
			foreach(c in companies)
			{
				if(congestion_policy == CONGESTION_GROW_REDUCED)
					GSNews.Create(GSNews.NT_GENERAL, GSText(GSText.STR_TOWN_CONGESTED_GROW_REDUCED_NEWS, this.id), c);
				else
					GSNews.Create(GSNews.NT_GENERAL, GSText(GSText.STR_TOWN_CONGESTED_GROW_STOPPED_NEWS, this.id), c);
			}

			if(congestion_policy == CONGESTION_GROW_REDUCED)
				Helper.SetSign(GSTown.GetLocation(this.id), GSText(GSText.STR_TOWN_CONGESTED_GROW_REDUCED_SIGN), true);
			else
				Helper.SetSign(GSTown.GetLocation(this.id), GSText(GSText.STR_TOWN_CONGESTED_GROW_STOPPED_SIGN), true);

			this.congestion_goal_id = GSGoal.New(GSCompany.COMPANY_INVALID, GSText(GSText.STR_REDUCE_TOWN_CONGESTION_GOAL, this.id), GSGoal.GT_TOWN, this.id);
		}
		else
		{
			Helper.SetSign(GSTown.GetLocation(this.id), "", true);
		}

		this.is_congested = is_congested;
	}

}
