
# script_ghost_desc()
sub script_ghost_desc
{
return "Ghost";
}

sub script_ghost_uses
{
return ( "node", "proxy" );
}

sub script_ghost_longdesc
{
return "Ghost is an open source publishing platform which is beautifully designed, easy to use, and free for everyone.";
}

# script_ghost_versions()
sub script_ghost_versions
{
return ( "0.4.2" );
}

sub script_ghost_release
{
return 2;	# For new node version
}

sub script_ghost_category
{
return "Blog";
}

# script_ghost_depends(&domain, version)
# Make sure we are on Linux
sub script_ghost_depends
{
local ($d, $ver) = @_;
local @rv;
if ($gconfig{'os_type'} !~ /-linux$/) {
	push(@rv, "Node.js binaries are only available on Linux");
	}
return @rv;
}

# script_ghost_params(&domain, version, &upgrade-info)
# Returns HTML for table rows for options for installing PHP-NUKE
sub script_ghost_params
{
local ($d, $ver, $upgrade) = @_;
local $rv;
local $hdir = &public_html_dir($d, 1);
if ($upgrade) {
	# Options are fixed when upgrading
	local $dir = $upgrade->{'opts'}->{'dir'};
	$dir =~ s/^$d->{'home'}\///;
	$rv .= &ui_table_row("Install directory", $dir);
	}
else {
	# Show editable install options
	$rv .= &ui_table_row("Install sub-directory under <tt>$hdir</tt>",
			     &ui_opt_textbox("dir", &substitute_scriptname_template("ghost", $d), 30, "At top level"));
	}
return $rv;
}

# script_ghost_parse(&domain, version, &in, &upgrade-info)
# Returns either a hash ref of parsed options, or an error string
sub script_ghost_parse
{
local ($d, $ver, $in, $upgrade) = @_;
if ($upgrade) {
	# Options are always the same
	return $upgrade->{'opts'};
	}
else {
	local $hdir = &public_html_dir($d, 0);
	$in->{'dir_def'} || $in->{'dir'} =~ /\S/ && $in->{'dir'} !~ /\.\./ ||
		return "Missing or invalid installation directory";
	local $dir = $in->{'dir_def'} ? $hdir : "$hdir/$in->{'dir'}";
	return { 'dir' => $dir,
		 'path' => $in->{'dir_def'} ? "/" : "/$in->{'dir'}", };
	}
}

# script_ghost_check(&domain, version, &opts, &upgrade-info)
# Returns an error message if a required option is missing or invalid
sub script_ghost_check
{
local ($d, $ver, $opts, $upgrade) = @_;
if (-r "$opts->{'dir'}/bin/node") {
	return "Ghost appears to be already installed in the selected directory";
	}
$opts->{'mongrels'} ||= 1;
return undef;
}

# script_ghost_files(&domain, version, &opts, &upgrade-info)
# Returns a list of files needed by Rails, each of which is a hash ref
# containing a name, filename and URL
sub script_ghost_files
{
local ($d, $ver, $opts, $upgrade) = @_;
my $nodever = "0.10.28";
local $u = &backquote_command("uname -m");
local $arch = $u =~ /x86_64/ ? "x64" : "x86";
local @files = ( {
	   'name' => "node",
	   'file' => "node-v$nodever-linux-$arch.tar.gz",
	   'url' => "http://nodejs.org/dist/v$nodever/node-v$nodever-linux-$arch.tar.gz"
	   },
	   {
	   'name' => "source",
	   'file' => "ghost-$ver.zip",
	   'url' => "https://ghost.org/zip/ghost-$ver.zip"
	   } );
return @files;
}

# script_ghost_install(&domain, version, &opts, &files, &upgrade-info)
# Actually installs PhpWiki, and returns either 1 and an informational
# message, or 0 and an error
sub script_ghost_install
{
local ($d, $version, $opts, $files, $upgrade) = @_;
local ($out, $ex);

# Stop running server if upgrading
if ($upgrade) {
	&script_ghost_stop_server($d, $opts);
	}

# Extract tar file to temp dir and copy to target
local $temp = &transname();
local $err = &extract_script_archive($files->{'node'}, $temp, $d,
                                     $opts->{'dir'}, "node-*");
$err && return (0, "Failed to extract binaries : $err");
local $temp2 = &transname();
local $err = &extract_script_archive($files->{'source'}, $temp2, $d,
                                     $opts->{'dir'});
$err && return (0, "Failed to extract source : $err");

# Pick a free port
my $port;
if ($upgrade) {
	$port = $opts->{'port'};
	}
else {
	$port = &allocate_mongrel_port(undef, 1);
	$opts->{'port'} = $port;
	}

# Run the npm install
my $out = &run_as_domain_user($d, "cd $opts->{'dir'} && $opts->{'dir'}/bin/npm install --production 2>&1");
if ($?) {
	return (-1, "NPM install of Ghost failed : $out");
	}

# Fix the port in the config file
my $url = &script_path_url($d, $opts);
my $noslash = $url;
$noslash =~ s/\/$//;
my $cfile = "$opts->{'dir'}/config.js";
my $cfilesrc = "$opts->{'dir'}/config.example.js";
if (!-r $cfile) {
	&copy_source_dest_as_domain_user($d, $cfilesrc, $cfile);
	my $lref = &read_file_lines_as_domain_user($d, $cfile);
	foreach my $l (@$lref) {
		if ($l =~ /^(\s*)url:\s*'\S+'/) {
			$l = $1."url: '".$noslash."',";
			}
		elsif ($l =~ /^(\s*)port:\s*'\S+'/) {
			$l = $1."port: '".$port."',";
			}
		}
	&flush_file_lines_as_domain_user($d, $cfile);
	}

# Pick a PID file and start command
my $pidfile = "$opts->{'dir'}/node.pid";
$opts->{'pidfile'} = $pidfile;
my $cmd = &get_ghost_start_cmd($d, $opts);

# Create init script
my $userd = $d->{'parent'} ? &get_domain($d->{'parent'}) : $d;
if (&foreign_installed("init") && $userd && $userd->{'unix'} && !$upgrade) {
	my $killcmd = "pkill -9 -P `cat $pidfile`";
	&foreign_require("init");
	&init::enable_at_boot(
		"ghost-$d->{'dom'}-$port",
		"Start Ghost server for $d->{'dom'}",
		&command_as_user($userd->{'user'}, 0, $cmd),
		&command_as_user($userd->{'user'}, 0, $killcmd),
		);
	}

# Start up node on the script, as the domain user
&run_as_domain_user($d, $cmd);

if (!$upgrade) {
	# Configure Apache to proxy to it
	&setup_mongrel_proxy($d, $opts->{'path'}, $port);
	}

local $rp = $opts->{'dir'};
$rp =~ s/^$d->{'home'}\///;
local $adminurl = $url.'ghost';
return (1, "Ghost installation complete. Go to <a target=_blank href='$adminurl'>$adminurl</a> to create the admin account.", "Under $rp", $url);
}

# script_ghost_uninstall(&domain, version, &opts)
# Un-installs a Rails installation, by deleting the directory and database.
# Returns 1 on success and a message, or 0 on failure and an error
sub script_ghost_uninstall
{
local ($d, $version, $opts) = @_;

# Shut down node server, if running
&script_ghost_stop_server($d, $opts);

# Remove bootup script
&foreign_require("init");
my $name =  "ghost-$d->{'dom'}-$opts->{'port'}";
if (defined(&init::delete_at_boot)) {
	&init::delete_at_boot($name)
	}
else {
	&init::disable_at_boot($name);
	}

# Remove the contents of the target directory
local $derr = &delete_script_install_directory($d, $opts);
return (0, $derr) if ($derr);

# Remove proxy Apache config entry for /ghost
&delete_mongrel_proxy($d, $opts->{'path'});

return (1, "Ghost directory deleted.");
}

# script_ghost_stop(&domain, &sinfo)
# Stop running mongrel process
sub script_ghost_stop
{
local ($d, $sinfo) = @_;
my $opts = $sinfo->{'opts'};
&script_ghost_stop_server($d, $opts);
&foreign_require("init");
my $name =  "ghost-$d->{'dom'}-$opts->{'port'}";
if (defined(&init::delete_at_boot)) {
	&init::delete_at_boot($name)
	}
else {
	&init::disable_at_boot($name);
	}
}

sub script_ghost_start_server
{
local ($d, $opts) = @_;
my $cmd = &get_ghost_start_cmd($d, $opts);
&run_as_domain_user($d, $cmd, 1);
}

# Return the PID if the node server is running
sub script_ghost_status_server
{
local ($d, $opts) = @_;
my $pid;
if ($opts->{'pidfile'}) {
	$pid = &check_pid_file($opts->{'pidfile'});
	}
return $pid ? ( $pid ) : ( );
}

# script_ghost_stop_server(&domain, &opts)
# Kill the running Node server process
sub script_ghost_stop_server
{
local ($d, $opts) = @_;
&run_as_domain_user($d, "cd $opts->{'dir'} && $opts->{'dir'}/bin/npm stop");
if ($opts->{'pidfile'}) {
	my $pid = &check_pid_file($opts->{'pidfile'});
	if ($pid) {
		&run_as_domain_user($d, "pkill -9 -P $pid");
		}
	&unlink_file_as_domain_user($d, $opts->{'pidfile'});
	}
}

sub script_ghost_latest
{
return ( "https://ghost.org/download/",
	 "ghost-([0-9\\.]+)\\.zip" );
}

sub script_ghost_site
{
return 'http://ghost.org/';
}

sub get_ghost_start_cmd
{
my ($d, $opts) = @_;
my $cmd = "cd $opts->{'dir'} ; $opts->{'dir'}/bin/npm start >ghost.log 2>&1 </dev/null & echo \$! >$opts->{'pidfile'}";
}

1;

